// SPDX-License-Identifier: GPL-2.0
/* main.c */
#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "core/qt-gui.h"
#include "core/subsurfacestartup.h"
#include "core/color.h"
#include "core/qthelper.h"
#include "core/downloadfromdcthread.h"

#include <QStringList>
#include <QApplication>
#include <QLoggingCategory>
#include <QLocale>
#include <git2.h>

// Implementation of STP logging
#include "core/ssrf.h"
#ifdef ENABLE_STARTUP_TIMING
#include <QElapsedTimer>
#include <QMutex>
#include <QMutexLocker>
void log_stp(const char *ident, QString *buf)
{
	static bool firstCall = true;
	static QElapsedTimer stpDuration;
	static QString stpText;
	static QMutex logMutex;

	QMutexLocker l(&logMutex);

	if (firstCall) {
		firstCall = false;
		stpDuration.start();
	}
	if (ident)
		stpText += QString("STP ") \
					.append(QString::number(stpDuration.elapsed())) \
					.append(" ms, ") \
					.append(ident) \
					.append("\n");
	if (buf) {
		*buf += "---------- startup timer ----------\n";
		*buf += stpText;
	}
}
#endif // ENABLE_STARTUP_TIMING


int main(int argc, char **argv)
{
	LOG_STP("main starting");

	int i;
	QGuiApplication::setAttribute(Qt::AA_EnableHighDpiScaling);
	QLoggingCategory::setFilterRules(QStringLiteral("qt.bluetooth* = true"));

	// Start application
	new QApplication(argc, argv);
	LOG_STP("main Qt started");

	// and get comand line arguments
	QStringList arguments = QCoreApplication::arguments();

	subsurface_console_init();

	for (i = 1; i < arguments.length(); i++) {
		QString a = arguments.at(i);
		if (!a.isEmpty() && a.at(0) == '-') {
			parse_argument(qPrintable(a));
			continue;
		}
	}
	git_libgit2_init();
	LOG_STP("main git loaded");
	setup_system_prefs();
	if (QLocale().measurementSystem() == QLocale::MetricSystem)
		default_prefs.units = SI_units;
	else
		default_prefs.units = IMPERIAL_units;
	copy_prefs(&default_prefs, &prefs);
	fill_profile_color();
	fill_computer_list();

	parse_xml_init();
	LOG_STP("main xml parsed");
	taglist_init_global();
	LOG_STP("main taglist done");
	init_ui();
	LOG_STP("main init_ui done");
	if (prefs.default_file_behavior == LOCAL_DEFAULT_FILE)
		set_filename(prefs.default_filename);
	else
		set_filename(NULL);

	// some hard coded settings
	prefs.animation_speed = 0; // we render the profile to pixmap, no animations

	// always show the divecomputer reported ceiling in red
	prefs.redceiling = 1;

	init_proxy();

	LOG_STP("main call run_ui (continue in qmlmanager)");
	if (!quit)
		run_ui();
	exit_ui();
	taglist_free(g_tag_list);
	parse_xml_exit();
	subsurface_console_exit();
	free_prefs();
	return 0;
}
