// SPDX-License-Identifier: GPL-2.0
#ifndef DIVETRIPMODEL_H
#define DIVETRIPMODEL_H

#include "core/dive.h"
#include <QAbstractItemModel>

class DiveTripModel : public QAbstractItemModel {
	Q_OBJECT
public:
	enum Column {
		NR,
		DATE,
		RATING,
		DEPTH,
		DURATION,
		TEMPERATURE,
		TOTALWEIGHT,
		SUIT,
		CYLINDER,
		GAS,
		SAC,
		OTU,
		MAXCNS,
		TAGS,
		PHOTOS,
		BUDDIES,
		COUNTRY,
		LOCATION,
		COLUMNS
	};

	enum ExtraRoles {
		STAR_ROLE = Qt::UserRole + 1,
		DIVE_ROLE,
		TRIP_ROLE,
		DIVE_IDX,
		SELECTED_ROLE
	};
	enum Layout {
		TREE,
		LIST,
		CURRENT
	};

	static DiveTripModel *instance();
	Qt::ItemFlags flags(const QModelIndex &index) const;
	QVariant headerData(int section, Qt::Orientation orientation, int role = Qt::DisplayRole) const override;
	bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole) override;
	DiveTripModel(QObject *parent = 0);
	void setLayout(Layout layout);
	QVariant data(const QModelIndex &index, int role) const;
	int columnCount(const QModelIndex&) const;
	int rowCount(const QModelIndex &parent) const;
	QModelIndex index(int row, int column, const QModelIndex &parent) const;
	QModelIndex parent(const QModelIndex &index) const;
	void filterFinished();

	// Used for sorting. This is a bit of a layering violation, as sorting should be performed
	// by the higher-up QSortFilterProxyModel, but it makes things so much easier!
	bool lessThan(const QModelIndex &i1, const QModelIndex &i2) const;
signals:
	// The propagation of selection changes is complex.
	// The control flow of dive-selection goes:
	// Commands/DiveListNotifier ---(dive */dive_trip *)---> DiveTripModel ---(QModelIndex)---> DiveListView
	// i.e. The command objects send changes in terms of pointer-to-dives, which the DiveTripModel transforms
	// into QModelIndexes according to the current view (tree/list). Finally, the DiveListView transforms these
	// indexes into local indexes according to current sorting/filtering and instructs the QSelectionModel to
	// perform the appropriate actions.
	void selectionChanged(const QVector<QModelIndex> &indexes, bool select);
	void newCurrentDive(QModelIndex index);
private slots:
	void divesAdded(dive_trip *trip, bool addTrip, const QVector<dive *> &dives);
	void divesDeleted(dive_trip *trip, bool deleteTrip, const QVector<dive *> &dives);
	void divesChanged(dive_trip *trip, const QVector<dive *> &dives);
	void divesTimeChanged(dive_trip *trip, timestamp_t delta, const QVector<dive *> &dives);
	void divesMovedBetweenTrips(dive_trip *from, dive_trip *to, bool deleteFrom, bool createTo, const QVector<dive *> &dives);
	void divesSelected(dive_trip *trip, const QVector<dive *> &dives);
	void divesDeselected(dive_trip *trip, const QVector<dive *> &dives);
	void currentDiveChanged();
private:
	// The model has up to two levels. At the top level, we have either trips or dives
	// that do not belong to trips. Such a top-level item is represented by the "Item"
	// struct. Two cases two consider:
	// 1) If "trip" is non-null, then this is a dive-trip and the dives are collected
	// in the dives vector.  Note that in principle we could also get the dives in a
	// trip from the backend, but there they are collected in a linked-list, which is
	// quite inconvenient to access.
	// 2) If "trip" is null, this is a dive and dives is supposed to contain exactly
	// one element, which is the corresponding dive.
	//
	// Top-level items are ordered by timestamp. For dives, the core function
	// dive_less_than is used, which guarantees a stable ordering even in the
	// case of equal timestamps. For dives and trips, place dives before trips
	// in the case of an equal timestamp. For trips with equal timestamps, the
	// order is currently undefined. This is currently not a problem, because
	// the core doesn't have a list of sorted trips. But nevertheless something
	// to keep in mind.
	struct Item {
		dive_trip		*trip;
		std::vector<dive *>	 dives;			// std::vector<> instead of QVector for insert() with three iterators
		Item(dive_trip *t, const QVector<dive *> &dives);
		Item(dive_trip *t, dive *d);			// Initialize a trip with one dive
		Item(dive *d);					// Initialize a top-level dive
		bool isDive(const dive *) const;		// Helper function: is this the give dive?
		dive *getDive() const;				// Helper function: returns top-level-dive or null
		timestamp_t when() const;			// Helper function: start time of dive *or* trip
	};
	// Comparison function between dive and arbitrary entry
	static bool dive_before_entry(const dive *d, const Item &entry);

	// Access trips and dives
	int findTripIdx(const dive_trip *trip) const;
	int findDiveIdx(const dive *d) const;			// Find _top_level_ dive
	int findDiveInTrip(int tripIdx, const dive *d) const;	// Find dive inside trip. Second parameter is index of trip
	int findInsertionIndex(timestamp_t when) const;		// Where to insert item with timestamp "when"

	// Access trip and dive data
	static QVariant diveData(const struct dive *d, int column, int role);
	static QVariant tripData(const dive_trip *trip, int column, int role);

	// Select or deselect dives
	void changeDiveSelection(dive_trip *trip, const QVector<dive *> &dives, bool select);

	// Addition and deletion of dives
	void addDivesToTrip(int idx, const QVector<dive *> &dives);

	dive *diveOrNull(const QModelIndex &index) const;	// Returns a dive if this index represents a dive, null otherwise
	QPair<dive_trip *, dive *> tripOrDive(const QModelIndex &index) const;
								// Returns either a pointer to a trip or a dive, or twice null of index is invalid
								// null, something is really wrong
	void setupModelData();
	std::vector<Item> items;				// Use std::vector for convenience of emplace_back()
	Layout currentLayout;
};

#endif
